<?php

/**
 * @package         Google Structured Data Markup
 * @version         2.1.5 Free
 *
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2016 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

defined('_JEXEC') or die('Restricted Access');

/**
 *  Google Structured Data Integration Class
 */
class GSDIntegration
{
    /**
     *  Global Joomla Application Object
     *
     *  @var  object
     */
    protected $app;

    /**
     *  Front-end View Name
     *
     *  @var  object
     */
    protected $view;

    /**
     *  Plugin params
     *
     *  @var  object
     */
    protected $params;

    /**
     *  Current active item
     *
     *  @var  object
     */
    protected $item;

    /**
     *  Default State
     *
     *  @var  bool
     */
    protected $defaultState = false;

    /**
     *  Class constructor
     *
     *  @param  object  $params  Plugin Params
     */
    public function __construct($params)
    {
        $this->params = $params;
        $this->app    = JFactory::getApplication();
        $this->view   = $this->app->input->get('view');
    }

    /**
     *  Route to proper method based on current view
     *
     *  @return  object  
     */
    public function getData()
    {
        // Check if integration is enabled
        if (!$this->isEnabled())
        {
            $this->log("disabled");
            return;
        }

        // Check if we support the current view
        $method = "view" . ucfirst($this->view);
        if (!method_exists($this, $method))
        {
            $this->log($method . " not found");
            return;
        }

        return $this->$method();
    }

    /**
     *  Class Logger
     *
     *  @param   string  $str  The message to log
     *
     *  @return  void
     */
    public function log($str)
    {
        GSDHelper::log($this->name . ": " . $str);
    }

    /**
     *  Checks if integration is enabled
     *
     *  @return  boolean
     */
    protected function isEnabled()
    {
        return $this->params->get($this->name . ".enabled", $this->defaultState);
    }

    /**
     *  Returns the right Item Text description
     *  Supports the Joomla Content and K2 extension only
     *
     *  @return  string
     */
    public function getItemDescription()
    {
        $itemText    = isset($this->item->introtext) && !empty($this->item->introtext) ? $this->item->introtext : $this->item->fulltext;
        $snippet     = $this->item->gsd->get("gsd.snippet");
        $text        = $this->item->gsd->get("gsd." . $snippet . ".description", $itemText);

        return GSDHelper::makeTextSafe($text);
    }

    /**
     *  Returns the right Item Title
     *  Supports the Joomla Content and K2 extension only
     *
     *  @return  string
     */
    public function getItemTitle()
    {
        $snippet = $this->item->gsd->get("gsd.snippet");
        $title   = $this->item->gsd->get("gsd." . $snippet . ".headline", $this->item->title);

        return GSDHelper::makeTextSafe($title, 0);
    }

    /**
     *  Returns the right Item Image
     *
     *  @return  string
     */
    public function getItemImage()
    {
        $snippet = $this->item->gsd->get("gsd.snippet");
        $imageOption = $this->item->gsd->get("gsd.$snippet.image");

        switch ($imageOption)
        {
            // Custom uploaded image
            case '2':
                $image = $this->item->gsd->get("gsd.$snippet.imgupload");
                break;
            // Custom image URL
            case '3':
                $image = $this->item->gsd->get("gsd.$snippet.imgurl");
                break;
            // Default item's image
            default:
                $image = $this->getItemDefaultImage();
                break;
        }

        return $image;
    }

    /**
     *  Default View Item
     *  Supports the Joomla Content and K2 extension only
     *
     *  @return  string
     */
    public function defaultViewItem()
    {
        if (!$id = $this->app->input->getInt('id'))
        {
            return;
        }

        // Abort if we don't have a valid K2 item object
        if (!$this->prepareItem($id))
        {
            return;
        }

        $p = $this->item->gsd;

        // Check we have a valid snippet enabled
        if (!$selectedSnippet = $p->get("gsd.snippet", false))
        {
            return;
        }

        // Prepare common data
        $data = array(
            "snippet"     => $selectedSnippet,
            "title"       => $this->getItemTitle(),
            "description" => $this->getItemDescription(),
            "image"       => GSDHelper::absURL($this->getItemImage()),
            "url"         => JURI::current(),
            "custom"      => $p->get("gsd.customcode")
        );

        // Prepare snippet data
        switch ($selectedSnippet)
        {
            case "product":
                $snippet = array(
                    "price"        => $p->get("gsd.product.offerPrice"),
                    "brand"        => $p->get("gsd.product.brandName"),
                    "ratingValue"  => $p->get("gsd.product.ratingValue"),
                    "reviewCount"  => $p->get("gsd.product.reviewCount"),
                    "sku"          => $p->get("gsd.product.sku"),
                    "currency"     => $p->get("gsd.product.currency"),
                    "condition"    => $p->get("gsd.product.offerItemCondition"),
                    "availability" => $p->get("gsd.product.offerAvailability")
                );
                break;
            case "article":
                $snippet = array(
                    "ratingValue" => $p->get("gsd.article.ratingValue"),
                    "reviewCount" => $p->get("gsd.article.reviewCount"),
                    "author"      => array("name" => JFactory::getUser($this->item->created_by)->name),
                    "publisher"   => array("name" => GSDHelper::getSiteName(), "logo" => GSDHelper::getSiteLogo()),
                    "created"     => GSDHelper::date($this->item->created),
                    "modified"    => GSDHelper::date($this->item->modified)
                );
                break;
            case "event":
                $startDate          = $p->get("gsd.event.startDate") . ' ' . $p->get("gsd.event.startTime");
                $endDate            = $p->get("gsd.event.endDate") . ' ' . $p->get("gsd.event.endTime");
                $offerStartDateTime = $p->get("gsd.event.offerStartDate") . ' ' . $p->get("gsd.event.offerStartTime");

                $snippet = array(
                    "type"      => $p->get("gsd.event.type"),
                    "starttime" => $p->get("gsd.event.startTime"),
                    "startdate" => GSDHelper::date($startDate),
                    "enddate"   => GSDHelper::date($endDate),
                    "location"  => array("name" => $p->get("gsd.event.locationName"), "address" => $p->get("gsd.event.locationAddress")),
                    "performer" => array("name" => $p->get("gsd.event.performerName"), "type" => $p->get("gsd.event.performerType")),
                    "status"    => $p->get("gsd.event.status"),
                    "offer"     => array(
                        "availability"   => $p->get("gsd.event.offerAvailability"), 
                        "startDateTime"  => GSDHelper::date($offerStartDateTime),
                        "price"          => $p->get("gsd.event.offerPrice"),
                        "currency"       => $p->get("gsd.event.offerCurrency"),
                        "inventoryLevel" => $p->get("gsd.event.offerInventoryLevel")
                    )
                );
                break;
            case "recipe":
                $snippet = array(
                    "authorName"    => JFactory::getUser($this->item->created_by)->name,
                    "datePublished" => JFactory::getDate($this->item->publish_up)->toISO8601(),
                    "dateCreated"   => JFactory::getDate($this->item->created)->toISO8601(),
                    "dateModified"  => JFactory::getDate($this->item->modified)->toISO8601(),
                    "prepTime"      => "PT" . $p->get("gsd.recipe.prepTime") . "M",
                    "cookTime"      => "PT" . $p->get("gsd.recipe.cookTime") . "M",
                    "totalTime"     => "PT" . $p->get("gsd.recipe.totalTime") . "M",
                    "calories"      => $p->get("gsd.recipe.calories"),
                    "yield"         => $p->get("gsd.recipe.yield"),
                    "ingredient"    => $this->makeArrayFromNewLine($p->get("gsd.recipe.ingredient")),
                    "instructions"  => $this->makeArrayFromNewLine($p->get("gsd.recipe.instructions")),
                    "ratingValue"   => $p->get("gsd.recipe.ratingValue"),
                    "reviewCount"   => $p->get("gsd.recipe.reviewCount")
                );
                break;
            case "course":
                $snippet = array(
                    "siteurl"   => GSDHelper::getSiteURL(),
                    "sitename"  => GSDHelper::getSiteName()
                );
                break;
            case 'review':
                $snippet = array(
                    "authorName"        => JFactory::getUser($this->item->created_by)->name,
                    "datePublished"     => JFactory::getDate($this->item->publish_up)->toISO8601(),
                    "siteurl"           => GSDHelper::getSiteURL(),
                    "sitename"          => GSDHelper::getSiteName(),
                    "itemReviewedType"  => $p->get("gsd.review.itemReviewedType"),
                    "itemReviewedName"  => $p->get("gsd.review.itemReviewedName"),
                    "address"           => $p->get("gsd.review.address"),
                    "priceRange"        => $p->get("gsd.review.priceRange"),
                    "telephone"         => $p->get("gsd.review.telephone"),
                    "ratingValue"       => $p->get("gsd.review.ratingValue")
                );
                break;
            default:
                // We catch a default case here in order to avoid null $snippet array
                $snippet = array();
        }

        return array_merge($data, $snippet);
    }

    /**
     *  Split string into array on each new line character
     *
     *  @param   string  $str  The string to split
     *
     *  @return  array
     */
    private function makeArrayFromNewLine($str)
    {
        $array = preg_split("/\\r\\n|\\r|\\n/", $str);

        if (!$array)
        {
            return $str;
        }

        return array_filter($array);
    }
}

?>