import java.io.*;
import java.util.*;

/**
 * This class replaces the previous C cleaner program. It does the following: 
 *  
 * Given a file on stdin 
 *  
 * 1) remove blank lines. 
 * 2) If a line with a ; by itself is found, that semi-colon is put on the previous non-blank line. 
 * 3) Remove lines that start with SQL comment characters -- 
 * 4) Convert all tabs to spaces. 
 * 5) replace two or more consecutive whitespace characters with a single space. 
 * 6) Remove trailing white space. 
 *  
 * If max.line.length is defined, then the code will write the maximum line length 
 * of any statement to be that value, and pad the value to pad.line.length. This is 
 * useful for DB2 running under Zos which has some really bizarre formatting rules. 
 *  
 * write the output to stdout. 
 * 
 * @author gsexton (3/30/2013)
 */
public class cleaner {

	private static int max_line_length=Integer.MAX_VALUE;
	private static int pad_line_length=Integer.MIN_VALUE;

	private static void ShowUsage(){
		System.out.println("Usage: cat sql | java -classpath . cleaner > out.file ");
	}

	private static void writeWrappedLine(String s) {
		
		StringBuilder sb=new StringBuilder(s);
		boolean bBreak=false;
		do {
			bBreak=false;
			boolean inQuote=false;
			int iLastWS=-1,iPos=0;
			for (int i=0; i < sb.length(); i++) {
				char c=sb.charAt(i);
				if (c=='\r') {
					// we're consuming them;
					c=' ';
					sb.setCharAt(i,' ');
				}
				if (c=='\n') {
					iPos=-1;
				} else if (Character.isWhitespace(c) && !inQuote) {
					iLastWS=i;
				} else if (c=='\'') {
					if (inQuote) {
						inQuote=false;
					} else {
						inQuote=true;
					}
				} else {
					if (iPos>=max_line_length) {
						if (iLastWS==-1) {
							System.err.println("Warning: Line: "+s+" no white space found for break insertion.");
							/*
							System.err.println(s);
							System.err.println("sb="+sb.toString());
							System.err.println("c='"+c+"]");
							System.err.println("i="+i+"\niPos="+iPos);
							*/
						} else {
							
							sb.setCharAt(iLastWS,'\n');
							bBreak=true;
						}
						
						
						break;
					
					}
				}
				iPos++;

			}
		} while (bBreak);
		sb.append('\n');
		int iPos=0;
		for (int iChar=0; iChar<sb.length(); iChar++) {
			char c=sb.charAt(iChar);
			if (c=='\n') {
				for (int iPad=0; iPad<(pad_line_length-iPos); iPad++) {
					System.out.print(' ');
				}
				iPos=-1;
			}
			System.out.print(c);
			iPos++;
		}
	}

	private static void writeOutputLine(String sLine) {
		if (max_line_length==Integer.MAX_VALUE) {
			System.out.println(sLine);
		} else {
			writeWrappedLine(sLine);
		}
	}

	public static void main(String[] argv) throws IOException {
		if (argv.length>0) {
			ShowUsage();
			System.exit(1);
		}
		String s=System.getProperty("max.line.length",null);
		if (s!=null) {
			max_line_length=Integer.valueOf(s);
			pad_line_length=Integer.valueOf(System.getProperty("pad.line.length","80"));
		}

		BufferedReader br=new BufferedReader(new InputStreamReader(System.in,"UTF-8"));
		int iRead;
		List<String> lst=new ArrayList<String>();
		String sLine,sTrimmed;
		while ((sLine=br.readLine())!=null) {
			sLine=sLine.replaceAll("\t"," ").replaceAll("\\s{2,}"," ").replaceAll("\\s+$","");

			sTrimmed=sLine.trim();
			if (sTrimmed.length()==0 || sTrimmed.startsWith("--")) {
				continue;
			} else if (sTrimmed.equals(";")) {
				if (lst.size()>0) {
					lst.set(lst.size()-1,lst.get(lst.size()-1)+sTrimmed);
				}
				continue;
			} else {
				lst.add(sLine);
			}
		}
		for (String sOut : lst) {
			writeOutputLine(sOut);
		}

	}

}
